<?php

namespace App\Http\Controllers\Admin;

use App\Helpers\TranslationHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\CharitableWork\CharitableWorkRequest;
use App\Models\CharitableWork;
use App\Models\Category;
use App\Models\Tag;
use App\Services\CharitableWorkService;
use App\ViewModels\CharitableWorkViewModel;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Yajra\DataTables\DataTables;

class CharitableWorkController extends Controller
{
    private $charitableWork;

    public function __construct()
    {
        $this->middleware(['auth:admin', 'permission:charitableWorks.view_all'], ['only' => ['index']]);
        $this->middleware(['auth:admin', 'permission:charitableWorks.view_details'], ['only' => ['show']]);
        $this->middleware(['auth:admin', 'permission:charitableWorks.create'], ['only' => ['create', 'update']]);
        $this->middleware(['auth:admin', 'permission:charitableWorks.edit'], ['only' => ['edit', 'store']]);
        $this->middleware(['auth:admin', 'permission:charitableWorks.delete'], ['only' => ['destroy']]);
        $this->middleware(['auth:admin', 'permission:charitableWorks.sort'], ['only' => ['reorder', 'saveReorder']]);
        $this->charitableWork = new CharitableWorkService();
    }
    
    /**
     * Display a listing of the charitableWork.
     */
    public function index(Request $request): View
    {
        return view('admin.pages.charitableWork.index');
    }


    public function table(DataTables $dataTables, Request $request)
    {
        $model = CharitableWork::ordered();

        return $dataTables->eloquent($model)->addIndexColumn()
            ->filter(function ($query) use ($request) {
                if ($search = $request->get('search')['value']) { 
                    $query->where(function ($q) use ($search) {
                        $q->where('name', 'LIKE', "%$search%");
                    });
                }
            })
            ->editColumn('id', function (CharitableWork $charitableWork) {
                return $charitableWork->id ?? '-';
            })
            ->editColumn('name', function (CharitableWork $charitableWork) {
                return $charitableWork->name;
            })
            ->editColumn('category', function (CharitableWork $charitableWork) {
                return $charitableWork->categories->name ?? '-'; 
            })
            ->editColumn('video', function (CharitableWork $charitableWork) {
                $currentLang = app()->getLocale(); 
                $mediaKey = 'charitableWork_video_one_' . $currentLang;

                $videoUrl = $charitableWork->getFirstMediaUrl($mediaKey);

                if ($videoUrl) {
                    return "<video width='150' height='100' controls>
                            <source src='{$videoUrl}' type='video/mp4'>
                            Your browser does not support the video tag.
                        </video>";
                } else {
                    return "لا يوجد فيديو";
                }
            })


            ->editColumn('created_at', function (CharitableWork $charitableWork) {
                return $charitableWork->created_at->format('d-m-Y h:i A') ?? '-';
            })
            ->addColumn('action', function (CharitableWork $charitableWork) {
                return view('admin.pages.charitableWork.buttons', compact('charitableWork'));
            })
            ->rawColumns(['video', 'action']) 
            ->startsWithSearch()
            ->make(true);
    }


    /**
     * Display a Single Row of the resource.
     */
    public function Show($id): View
    {

        $charitableWork = CharitableWork::find($id);
        return view('admin.pages.charitableWork.show', get_defined_vars());
    }

    /**
     * Show the form for creating a new charitableWork.
     */
    public function create(): View
    {
        $tags = Tag::get();
        return view('admin.pages.charitableWork.form',  new CharitableWorkViewModel(), get_defined_vars());
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CharitableWorkRequest $request)
    {
        // return $request;
        $this->charitableWork->createCharitableWork($request->validated());
        session()->flash('success', TranslationHelper::translate('Operation Success'));
        return back();
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(CharitableWork $charitableWork): View
    {
       
        return view('admin.pages.charitableWork.form',  new CharitableWorkViewModel($charitableWork), get_defined_vars());
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(CharitableWorkRequest $request, CharitableWork $charitableWork): RedirectResponse
    {
        //        return $request;
        $this->charitableWork->updateCharitableWork($charitableWork, $request->validated());
        session()->flash('success', TranslationHelper::translate('Successfully Updated'));
        return back();
    }
   

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(CharitableWork $charitableWork): JsonResponse
    {
        $this->charitableWork->deleteCharitableWork($charitableWork);
        return response()->json(['status' => 'success', 'message' => TranslationHelper::translate('Successfully Deleted')]);
    }

    public function reorder(CharitableWork $charitableWork)
    {
        return $this->charitableWork->reorder($charitableWork, 'name', 'admin.pages.charitableWork.reorder', 1);
    }

    public function saveReorder(Request $request, CharitableWork $charitableWork)
    {
        $all_entries = $request->input('tree');
        return $this->charitableWork->saveReorder($all_entries, $charitableWork);
    }

}
